#include "ringbuf.h"

#include <stdlib.h>
#include <stdbool.h>

/**
 * Get the number of full entries in the ringbuffer
 */
int getNumFull(ringbuffer_t *ringbuffer)
{
    return (ringbuffer->nextEmptyIndex - ringbuffer->startIndex + 2 * RINGBUFFER_SIZE) % RINGBUFFER_SIZE;
}

/**
 * Get the capacity left in the buffer
 */
int getNumEmpty(ringbuffer_t *ringbuffer)
{
    return RINGBUFFER_SIZE - getNumFull(ringbuffer);
}

bool bufCanWrite(ringbuffer_t *ringbuffer)
{
    return getNumEmpty(ringbuffer) > 0;
}

bool bufCanRead(ringbuffer_t *ringbuffer)
{
    return getNumFull(ringbuffer) > 0;
}

/**
 * Write single character to ringbuffer if there is space left
 * @param ringbuffer Buffer to use
 * @param value Value to write
 * @param critical bool if the section is critical (irq disable)
 * @return if the character was written
 */
bool writeRingbufferChar(ringbuffer_t *ringbuffer, char value, bool critical)
{
    if (critical)
    {
        __disable_irq();
    }

    bool canWrite = bufCanWrite(ringbuffer);
    if (canWrite)
    {
        ringbuffer->buffer[ringbuffer->nextEmptyIndex] = value;
        ringbuffer->nextEmptyIndex = (ringbuffer->nextEmptyIndex + 1) % RINGBUFFER_SIZE;
    }

    if (critical)
    {
        __enable_irq();
    }
    return canWrite;
}

/**
 * Write the data into the ringbuffer until:
 *      - \n is reached
 *      - \0 is reached
 *      - The ringbuffer is full
 * @param ringbuffer Buffer to use
 * @param data Data to write
 * @param critical bool if the section is critical (irq disable)
 * @return if the data was written until \n or \0
 */
bool writeRingbufferLine(ringbuffer_t *ringbuffer, char *data, bool critical)
{
    if (critical)
    {
        __disable_irq();
    }

    int16_t i = 0;
    for (; bufCanWrite(ringbuffer) && data[i] != '\0' && data[i] != '\n'; i++)
    {
        writeRingbufferChar(ringbuffer, data[i], false); // not critical, since the interrupts are already disabled if it was critical
    }

    if (bufCanWrite(ringbuffer) && data[i] != 0)
    {
        writeRingbufferChar(ringbuffer, data[i], false); // also write the \n character
    }

    bool toReturn = data[i] == '\0' || data[i] == '\n';

    if (critical)
    {
        __enable_irq();
    }
    return toReturn;
}

/**
 * Write the data into the ringbuffer until:
 *      - The ringbuffer is full
 *      - length many characters were written
 * @param ringbuffer Buffer to use
 * @param data Data to write
 * @param length Num of chars to write
 * @param critical bool if the section is critical (irq disable)
 * @return if the data was written until \n or \0
 */
bool writeRingbufferLen(ringbuffer_t *ringbuffer, char *data, int length, bool critical)
{
    if (critical)
    {
        __disable_irq();
    }
    int16_t i = 0;
    for (; i < length && bufCanWrite(ringbuffer); i++)
    {
        writeRingbufferChar(ringbuffer, data[i], false); // not critical, since the interrupts are already disabled if it was critical
    }

    if (critical)
    {
        __enable_irq();
    }
    return (i == length);
}

/**
 * Read single character from ringbuffer if there is one left
 * @param ringbuffer Buffer to read from
 * @param dst Buffer to read to
 * @param critical bool if the section is critical (irq disable)
 * @return amount of characters read
 */
int readRingbufferChar(ringbuffer_t *ringbuffer, char *dst, bool critical)
{
    if (critical)
    {
        __disable_irq();
    }

    bool canRead = bufCanRead(ringbuffer);
    if (canRead)
    {
        *dst = ringbuffer->buffer[ringbuffer->startIndex];
        ringbuffer->startIndex = (ringbuffer->startIndex + 1) % RINGBUFFER_SIZE;
    }

    if (critical)
    {
        __enable_irq();
    }
    return canRead;
}

/**
 * Read the data from the ringbuffer until:
 *      - \n is reached
 *      - \0 is reached
 *      - The ringbuffer is empty
 * @param ringbuffer Buffer to read from
 * @param dst Buffer to read to
 * @param critical bool if the section is critical (irq disable)
 * @return amount of characters read
 */
int readRingbufferLine(ringbuffer_t *ringbuffer, char *dst, bool critical)
{
    if (critical)
    {
        __disable_irq();
    }

    int16_t i = 0;
    for (; bufCanRead(ringbuffer) && ringbuffer->buffer[ringbuffer->startIndex] != '\0' && ringbuffer->buffer[ringbuffer->startIndex] != '\n'; i++)
    {
        readRingbufferChar(ringbuffer, dst + i, false); // not critical, since the interrupts are already disabled if it was critical
    }

    if (bufCanRead(ringbuffer))
    {
        readRingbufferChar(ringbuffer, dst + i, false); // also read the \0 or \n byte if availible
        i++;
    }

    if (critical)
    {
        __enable_irq();
    }

    return i;
}

/**
 * Write the data into the ringbuffer until:
 *      - The ringbuffer is full
 *      - length many characters were written
 * @param ringbuffer Buffer to read from
 * @param dst Buffer to read to
 * @param length Num of chars to read
 * @param critical bool if the section is critical (irq disable)
 * @return amount of characters read
 */
int readRingbufferLen(ringbuffer_t *ringbuffer, char *dst, int length, bool critical)
{
    if (critical)
    {
        __disable_irq();
    }

    int16_t i = 0;
    for (; bufCanRead(ringbuffer) && i < length; i++)
    {
        readRingbufferChar(ringbuffer, dst + i, false); // not critical, since the interrupts are already disabled if it was critical
    }

    if (critical)
    {
        __enable_irq();
    }
    return i;
}
