#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>

#include "main.h"
#include "spi.h"
#include "uart.h"
#include "board.h"
#include "mapping.h"
#include "nvic.h"
#include "bitops.h"

spi_data_t spi_data;

void initSPIdata(void)
{
    memset(&spi_data, 0, sizeof(spi_data_t));
}

void pinConfigSPI(void)
{
    // Enable Clock for GPIOA, GPIOB and SPI1
    LL_AHB1_GRP1_EnableClock(LL_AHB1_GRP1_PERIPH_GPIOA); // in fact not necessary, as we already did this in pinConfigUart
    LL_AHB1_GRP1_EnableClock(LL_AHB1_GRP1_PERIPH_GPIOB);
    LL_APB2_GRP1_EnableClock(LL_APB2_GRP1_PERIPH_SPI1);

    /**
     * GPIO A pins for Board communication
     * PA12 - Enable LEDs/IR-Sensoren
     * PA15 - Schalter MUX/DEMUX
     * PA4  - Enable Servos
     *
     * 15:
     *  0 -> connect LED/IR
     *  1 -> connect ServoHost
     *
     * \see doc/Bonnie.pdf 2.1:
     *
     * Um die SPI des STM32 mit der LED-
     * Anzeige und den Lichtschranken zu verbinden,
     * muss der Pin PA15 logisch ’0’sein
     *
     * Durch eine positive Flanke an Pin PA12 übernehmen die LED-Controller (74HC595D in Abbildung 2)
     * die Werte der jeweiligen Register in ihre Ausgangslatches.
     * Wenn an Pin PA12 des Mikrocontrollers logisch ’1’ anliegt,
     * können die Daten durch die Register durchgeschoben werden, bei logisch ’0’
     * werden die Zustände der Infrarotempfangsdioden in die Register übernommen
     *
     * Error in 2.2 Aktuatoren:
     * Um die SPI des STM32 mit der LED-Anzeige und den Lichtschranken zu verbinden,
     * muss der Pin PA15 logisch ’1’ sein.
     * Anstatt: "LED-Anzeige und den Lichtschranken" -> "ServoHost"
     **/

    // Setup the controlling GPIO pins
    LL_GPIO_InitTypeDef GPIO_InitStruct = {0};
    LL_GPIO_StructInit(&GPIO_InitStruct);

    GPIO_InitStruct.Pin = LL_GPIO_PIN_4 | LL_GPIO_PIN_12 | LL_GPIO_PIN_15;
    GPIO_InitStruct.Mode = LL_GPIO_MODE_OUTPUT;
    GPIO_InitStruct.Speed = LL_GPIO_SPEED_FREQ_HIGH;
    GPIO_InitStruct.OutputType = LL_GPIO_OUTPUT_PUSHPULL;
    GPIO_InitStruct.Pull = LL_GPIO_PULL_NO;
    LL_GPIO_Init(GPIOA, &GPIO_InitStruct);

    /**
     * GPIOB pins for SPI communication
     * PB3 - SCLK (clock)
     * PB4 - MISO (master in slave out)
     * PB5 - MOSI (master out slave in)
     *
     * (we are the master)
     */

    // Configure GPIO Pins for SPI
    LL_GPIO_StructInit(&GPIO_InitStruct);

    GPIO_InitStruct.Pin = LL_GPIO_PIN_3 | LL_GPIO_PIN_4 | LL_GPIO_PIN_5;
    GPIO_InitStruct.Mode = LL_GPIO_MODE_ALTERNATE;
    GPIO_InitStruct.Speed = LL_GPIO_SPEED_FREQ_HIGH;
    GPIO_InitStruct.OutputType = LL_GPIO_OUTPUT_PUSHPULL;
    GPIO_InitStruct.Pull = LL_GPIO_PULL_NO;
    GPIO_InitStruct.Alternate = LL_GPIO_AF_5; // SPI1 alternate function (Chip manual p. 42)
    LL_GPIO_Init(GPIOB, &GPIO_InitStruct);

    // Configure SPI
    LL_SPI_InitTypeDef SPI_InitStruct = {0};
    LL_SPI_StructInit(&SPI_InitStruct);

    SPI_InitStruct.TransferDirection = LL_SPI_FULL_DUPLEX;
    SPI_InitStruct.Mode = LL_SPI_MODE_MASTER;                      // Master mode
    SPI_InitStruct.DataWidth = LL_SPI_DATAWIDTH_8BIT;              // Data width: 8 bits (16bit is just two 8bit sends)
    SPI_InitStruct.ClockPolarity = LL_SPI_POLARITY_LOW;            // clock line idle low
    SPI_InitStruct.ClockPhase = LL_SPI_PHASE_1EDGE;                // Clock phase first edge
    SPI_InitStruct.BitOrder = LL_SPI_MSB_FIRST;                    // MSB first
    SPI_InitStruct.CRCCalculation = LL_SPI_CRCCALCULATION_DISABLE; // No CRC calculation
    SPI_InitStruct.NSS = LL_SPI_NSS_SOFT;                          // We set it via software

    // APB2 has prescaler 2 -> 64k/2 = 32 k
    // We divide by 256 -> 32k/256 = 125
    SPI_InitStruct.BaudRate = LL_SPI_BAUDRATEPRESCALER_DIV256;

    LL_SPI_Init(SPI1, &SPI_InitStruct);

    LL_SPI_Enable(SPI1);

		// RXNE works like a "transfer complete" flag/interrupt
    LL_SPI_EnableIT_RXNE(SPI1); // interrupt on receive <- NSS=SOFT necessary

    uint32_t encoded_priority = NVIC_EncodePriority(NVIC_PRIORITY_GROUP, 1, 0);
    NVIC_SetPriority(SPI1_IRQn, encoded_priority);

    initSPIdata();
}

uint8_t ir_data[4];
spi_state_t spiState = SPI_STATE_LEDIR_INIT;
uint8_t spiSide = 3;

// 16 LED pro Seite, 8 IR pro Seite
// 8 LED
// 8 IR
// 8 LED

void spiStateMachine(bool reset)
{
    if (reset)
    {
        spiState = SPI_STATE_LEDIR_INIT;
    }

    switch (spiState)
    {
    case SPI_STATE_LEDIR_INIT:
				NVIC_EnableIRQ(SPI1_IRQn); // only accept SPI interrupts while state-machine is running

        LL_GPIO_ResetOutputPin(GPIOA, LL_GPIO_PIN_15); // Set Pin 15 to 0 to connect to LED/IR
        LL_GPIO_SetOutputPin(GPIOA, LL_GPIO_PIN_12);   // Set Pin 12 to 1 to shift the data through the registers
        memset(&ir_data, 0, 4 * sizeof(uint8_t));      // reset memset

        // the timer is for starting the state machine
        // -> disable it for as long as the state machine is running
        NVIC_DisableIRQ(TIM2_IRQn);

        spiSide = 3;
        ++spiState;
        // fallthrough

    case SPI_STATE_LEDIR_WAIT1:
        if (spiSide < 3)
        {
            LL_SPI_ReceiveData8(SPI1); // discard received data from the LEDs
        }
        LL_SPI_TransmitData8(SPI1, U16_HIGH_BYTE(spi_data.leds[spiSide]));
        break;

    case SPI_STATE_LEDIR_WAIT2:
        LL_SPI_ReceiveData8(SPI1);     // discard received data from the LEDs
				LL_SPI_TransmitData8(SPI1, 0); // NOTE: Send nothing to the IR registers
        break;

    case SPI_STATE_LEDIR_WAIT3:
        ir_data[spiSide] = LL_SPI_ReceiveData8(SPI1);
        LL_SPI_TransmitData8(SPI1, U16_LOW_BYTE(spi_data.leds[spiSide]));
        if (spiSide > 0)
        {
            --spiSide;
            spiState = SPI_STATE_LEDIR_WAIT1;
            return;
        }
        break;

    case SPI_STATE_LEDIR_LAST:
        LL_SPI_ReceiveData8(SPI1); // discard received data from the LEDs

        spi_data.ir_receive[0] = interleaveBits((ir_data[2]), reverseBitorder8(ir_data[0]));
        spi_data.ir_receive[1] = reverseBitorder16(interleaveBits((ir_data[1]), reverseBitorder8(ir_data[3])));
				mapIRtoLED(); // Map the IR data to the LEDs

        // transmission with LEDs/IR-sensors is finished
        LL_GPIO_ResetOutputPin(GPIOA, LL_GPIO_PIN_12);

				++spiState;
				// fallthrough

    case SPI_STATE_SERVOS_INIT:
        LL_GPIO_SetOutputPin(GPIOA, LL_GPIO_PIN_15);  // Set Pin 15 to 1 to connect to the servo hub
        LL_GPIO_ResetOutputPin(GPIOA, LL_GPIO_PIN_4); // Set Pin 4 to low to enable servo hubs receive

        ++spiState;
        // falltrough

    case SPI_STATE_SERVOS_TXH1:
        LL_SPI_TransmitData8(SPI1, U16_HIGH_BYTE(spi_data.servos[0])); // send upper byte first
        break;

    case SPI_STATE_SERVOS_TXL1:
				LL_SPI_ReceiveData8(SPI1); // discard received data from the LEDs
        LL_SPI_TransmitData8(SPI1, U16_LOW_BYTE(spi_data.servos[0])); // send lower byte
        break;

    case SPI_STATE_SERVOS_TXH2:
				LL_SPI_ReceiveData8(SPI1); // discard received data from the LEDs
        LL_SPI_TransmitData8(SPI1, U16_HIGH_BYTE(spi_data.servos[1])); // send upper byte first
        break;

    case SPI_STATE_SERVOS_TXL2:
				LL_SPI_ReceiveData8(SPI1); // discard received data from the LEDs
        LL_SPI_TransmitData8(SPI1, U16_LOW_BYTE(spi_data.servos[1])); // send lower byte
        break;

    case SPI_STATE_SERVOS_LAST:
        LL_SPI_ReceiveData8(SPI1); // discard received data from the LEDs

        LL_GPIO_SetOutputPin(GPIOA, LL_GPIO_PIN_4); // Set Pin 4 to high for the server hub to save the data

        NVIC_DisableIRQ(SPI1_IRQn); // we only want/need SPI interrupts during the state machine
        NVIC_EnableIRQ(TIM2_IRQn);  // disabled as long as state machine is running -> disable now
        break;

    case SPI_STATE_FIN:
        // should never arrive here
        return; // we never go further than the fin state

    default:
        // unknown state -> this should never happen
        return;
    }
    spiState++;
}

void SPI1_IRQHandler(void)
{
    spiStateMachine(false); // false -> no reset
}