/*pint config */
#include "main.h"
#include "nvic.h"
#include "uart.h"
#include "pc_commands.h"

ringbuffer_t uartBuffer;

void pinConfigUart(void)
{
    /**
     * GPIO A pins for UART Communication
     * PA2 - TX
     * PA3 - RX
     * For these pins UART2 need to be used
     **/

    LL_AHB1_GRP1_EnableClock(LL_AHB1_GRP1_PERIPH_GPIOA);

    LL_GPIO_InitTypeDef GPIO_InitStructure;
    LL_USART_InitTypeDef USART_InitStruct;

    LL_GPIO_StructInit(&GPIO_InitStructure);

    GPIO_InitStructure.Mode = LL_GPIO_MODE_ALTERNATE;
    GPIO_InitStructure.OutputType = LL_GPIO_OUTPUT_PUSHPULL;
    GPIO_InitStructure.Pin = LL_GPIO_PIN_2 | LL_GPIO_PIN_3;
    GPIO_InitStructure.Pull = LL_GPIO_PULL_NO;
    GPIO_InitStructure.Speed = LL_GPIO_SPEED_FREQ_HIGH;
    GPIO_InitStructure.Alternate = LL_GPIO_AF_7;

    LL_GPIO_Init(GPIOA, &GPIO_InitStructure);

    /**
     * USART2 Setup
     * Baud: 9600, 8 bit words, stopbit, no parity bit
     */
		LL_APB1_GRP1_EnableClock(LL_APB1_GRP1_PERIPH_USART2);

    LL_USART_StructInit(&USART_InitStruct);

    USART_InitStruct.BaudRate = 9600;                               // Baudrate = 9600
    USART_InitStruct.DataWidth = LL_USART_DATAWIDTH_8B;             // 8 bit Daten
    USART_InitStruct.StopBits = LL_USART_STOPBITS_1;                // Stopbits = 1
    USART_InitStruct.Parity = LL_USART_PARITY_NONE;                 // kein Paritybit
    USART_InitStruct.HardwareFlowControl = LL_USART_HWCONTROL_NONE; // keine Hardware Flow Control
    USART_InitStruct.TransferDirection = LL_USART_DIRECTION_TX_RX;  // Einstellungen gelten für RX und TX, wir wollen beide Richtungen aktiviere

    LL_USART_Init(USART2, &USART_InitStruct);

    LL_USART_Enable(USART2);

    LL_USART_EnableIT_RXNE(USART2);

    /**
     * NVIC Setup
     * preemp prio: 1
     * sub prio: 0
     */
    uint32_t encoded_priority = NVIC_EncodePriority(NVIC_PRIORITY_GROUP, 1, 0);
    NVIC_SetPriority(USART2_IRQn, encoded_priority);

    NVIC_EnableIRQ(USART2_IRQn);
}

void initUartBuffer(void)
{
    // memset 0
    for (int i = 0; i < sizeof(uartBuffer.buffer); i++)
    {
        uartBuffer.buffer[i] = 0;
    }
    uartBuffer.nextEmptyIndex = 0;
    uartBuffer.startIndex = 0;
}

void writeUart(uint8_t value)
{
    writeRingbufferChar(&uartBuffer, value, true);
}

void writeUartLine(char *str)
{
    writeRingbufferLine(&uartBuffer, str, true);
}

void writeUartLen(char *str, int n)
{
    for (int i = 0; i < n; i++)
    {
        writeUart(str[i]);
    }
}

void flushUart(void)
{
    if (bufCanRead(&uartBuffer))
    {
				__disable_irq();
        // wait for transfer buffer to be empty -> sending process to finish
        while (LL_USART_IsActiveFlag_TXE(USART2) && bufCanRead(&uartBuffer))
        {
						char buf[1];
						readRingbufferChar(&uartBuffer, buf, false);
						LL_USART_TransmitData8(USART2, *buf);
        };
				__enable_irq();
        LL_USART_EnableIT_TXE(USART2); // enable interrupt for the next call
    }
    else
    {
        LL_USART_DisableIT_TXE(USART2);
    }
}

void USART2_IRQHandler(void)
{
    // if we received something
    while (LL_USART_IsActiveFlag_RXNE(USART2))
    {
        // only read if there is something to read
        processInput(LL_USART_ReceiveData8(USART2));
    }
    // if tansmit buffer is empty -> still space in transmit buffer
    flushUart();
}
